% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/years.R
\name{years}
\alias{years}
\title{Compute one of the life years measures}
\usage{
years(
  formula = formula(data),
  data,
  measure = c("yd", "yl2017", "yl2013"),
  ratetable = relsurv::slopop,
  rmap,
  var.estimator = c("none", "bootstrap", "greenwood"),
  B = 100,
  precision = 30,
  add.times,
  na.action = stats::na.omit,
  conf.int = 0.95,
  timefix = FALSE,
  admin.cens,
  arg.example = FALSE,
  cause.val,
  is.boot = FALSE,
  first.boot = FALSE
)
}
\arguments{
\item{formula}{a formula object, with the response as a \code{Surv} object
on the left of a \code{~} operator, and, \code{~1} specified on the right.

NOTE: The follow-up time must be in days.}

\item{data}{a data.frame in which to interpret the variables named in the
\code{formula}.}

\item{measure}{choose which measure is used: 'yd' (life years difference; Manevski, Ruzic Gorenjec, Andersen, Pohar Perme, 2022), 'yl2017' (years lost/saved; Andersen 2017),
'yl2013' (years lost/saved; Andersen 2013).
the population cumulative incidence curve. Relevant only for \code{measure='yd'}.
For \code{measure='yl2013'} and \code{measure='yl2017'} the estimators defined in (Andersen, 2013) and (Andersen, 2017)
are used.}

\item{ratetable}{a table of event rates, organized as a \code{ratetable}
object, such as \code{slopop}.}

\item{rmap}{an optional list to be used if the variables are not organized
and named in the same way as in the \code{ratetable} object. See details
below.}

\item{var.estimator}{Choose the estimator for the variance ('none', 'bootstrap', 'greenwood'). Default is 'none'.}

\item{B}{if \code{var.estimator} is 'bootstrap'. The number of bootstrap replications. Default is 100.}

\item{precision}{precision for numerical integration of the population curve. Default is 30 (days).
The value may be decreased to get a
higher precision or increased to achieve a faster calculation.}

\item{add.times}{specific times at which the curves should be reported.}

\item{na.action}{a missing-data filter function. Default is \code{na.omit}.}

\item{conf.int}{the confidence level for a two-sided confidence interval. Default is 0.95.}

\item{timefix}{the timefix argument in survival::survfit.formula. Default is FALSE.}

\item{admin.cens}{if a Date is supplied, administrative censoring is taken into account at that time
in the population curve. Works only if there's late entry, e.g. if the formula is \code{Surv(start,stop,event)~1}.}

\item{arg.example}{temporary argument, used for checking additionalities.}

\item{cause.val}{for competing risks, to be added.}

\item{is.boot}{if TRUE, the function \code{years} has been called during a bootstrap replication.}

\item{first.boot}{if TRUE, this is the first bootstrap replication.}
}
\value{
A list containing the years measure, the observed and population curves (or the excess curve for Andersen 2013).
The values are given as separate data.frames through time. Times are given in days, all areas are given in years.
Functions \code{plot_f} and \code{plot_years} can be then used for plotting.
}
\description{
Provides an estimate for one of the following measures: years lost (Andersen, 2013), years lost/saved (Andersen, 2017), or
life years difference (Manevski, Ruzic Gorenjec, Andersen, Pohar Perme, 2022).
}
\details{
The life years difference (\code{measure='yd'}) is taken by default. If other
measures are of interest, use the \code{measure} argument.

The follow-up time must be specified in days. The \code{ratetable}
being used may have different variable names and formats than the user's
data set, this is dealt with the \code{rmap} argument. For example, if
age is in years in the data but in days in the \code{ratetable} object,
age=age*365.241 should be used. The calendar year can be in any date format
(date, Date and POSIXt are allowed), the date formats in the
\code{ratetable} and in the data may differ.

Numerical integration is required for the population curves. The integration
precision is set with argument \code{precision}, which defaults to 30-day
intervals. For higher accuracy take a smaller value (e.g. precision=1 makes
the integration on a daily basis).

The observed curves are reported at event and censoring times. The
population curves are reported at all times used for the numerical integration.
Note that for the years lost (Andersen, 2013) measure, only the excess absolute risk is reported.
}
\examples{

library(relsurv)
# Estimate the life years difference for the rdata dataset.
mod <- years(Surv(time, cens)~1, data=rdata, measure='yd', ratetable=slopop,
             rmap=list(age=age*365.241), var.estimator = 'none')
# Plot the absolute risk (observed and population curve):
plot_f(mod)
# Plot the life years difference estimate:
plot_years(mod, conf.int=FALSE)
}
\seealso{
\code{plot_f}, \code{plot_years}
}
