/*
Copyright (C) 2002-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include "config.h"
#include <stdio.h>
#include <ctype.h>
#include "header.h"

void
printentryname(FILE *fout, const char *s)
{
  if (isdigit(*s))
    fprintf(fout,"p%s",s);
  else
    fprintf(fout,"%s",s);
}

void
printentry(FILE *fout, int x)
{
  const char *s=value[x].val.str;
  printentryname(fout, s);
}

extern int indent;
void printnodeparens(FILE *fout, int n, int parens)
{
  int i,v;
  context *fc;
  int x,y;
  if (n<0)
    return;
  x=tree[n].x;
  y=tree[n].y;
  gencomment(fout,n,!indent);
  if (debug>=3) fprintf(fout,"/*%s:%d:%s*/",GPname(tree[n].t),tree[n].m,funcname(tree[n].f));
  switch(tree[n].f)
  {
  case Fseq:
    printnode(fout,x);
    if (x>=0 && tree[x].f==Fseq) x=tree[x].y;
    if (x>=0 && tree[x].f!=Fblock && tree[x].f!=Fdeffunc)
    {
      if (indent)
        fprintf(fout,";\n");
      else
        fprintf(fout,"; \\\n");
      genindent(fout);
    }
    printnode(fout,y);
    break;
  case Fmatrix:
    fputc('[',fout);
    printnode(fout,x);
    if (y>0)
    {
      fprintf(fout,",");
      printnode(fout,y);
    }
    fputc(']',fout);
    break;
  case Frange:
    if (x!=GNORANGE)
    {
      printnode(fout,x);
      if (y!=GNORANGE)
      {
        fputs("..",fout);
        printnode(fout,y);
      }
    }
    else if (y!=GNORANGE)
    {
      fputc('^',fout);
      printnode(fout,y);
    }
    break;
  case Fnorange:
    break;
  case Fmatcoeff:
    printnodeparens(fout,x,1);
    printnode(fout,y);
    break;
  case Fassign:
    if (parens)
      fputc('(',fout);
    printnode(fout,x);
    fprintf(fout,"=");
    printnode(fout,y);
    if (parens)
      fputc(')',fout);
    break;
  case Fconst:
    switch(value[x].type)
    {
    case CSTsmall:
      fprintf(fout,"%ld",value[x].val.small);
      break;
    case CSTsmallreal:
      fprintf(fout,"%ld.",value[x].val.small);
      break;
    case CSTint: case CSTreal:
      fprintf(fout,"%s",value[x].val.str);
      break;
    case CSTstr:
      fprintf(fout,"\"%s\"",value[x].val.str);
      break;
    }
    break;
  case Fsmall:
    fprintf(fout,"%d",x);
    break;
  case Fmatrixelts:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fmatrixlines:
    printnode(fout,x);
    fprintf(fout,";");
    /*Would be nice to indent matrix, alas I have no way to know where is the
     * bottomrow*/
    printnode(fout,y);
    break;
  case Fmat:
  case Fvec:
    fprintf(fout,"[");
    if (tree[n].f==Fmat && x==-1)
      fputc(';',fout);
    else
      printnode(fout,x);
    fprintf(fout,"]");
    break;
  case Flistarg:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fentry:
    printentry(fout,x);
    break;
  case Ffun:
    fprintf(fout,"%s",lfunc[x].gpname);
    break;
  case Ffunction:
    if (x<OPnboperator || (value[x].val.str[0]=='_' && value[x].val.str[1]=='.'))
    {
      int arg[STACKSZ];
      int nb=genlistargs(n,arg,0,STACKSZ);
      int i=0;
      const char *p, *name=value[x].val.str;
      if (x==OPcompr)
      {
        fputc('[',fout);
        printnode(fout,arg[2]);
        fputc('|',fout);
        printnode(fout,arg[1]);
        fputs("<-",fout);
        printnode(fout,arg[0]);
        if (nb==4)
        {
          fputc(',',fout);
          printnode(fout,arg[3]);
        }
        fputc(']',fout);
      }
      else if (x==OPcoeff)
      {
        printnodeparens(fout,arg[0],1);
        fputc('[',fout);
        printnode(fout,arg[1]);
        if (nb==3)
        {
          fputc(',',fout);
          printnode(fout,arg[2]);
        }
        fputc(']',fout);
      }
      else
      {
        if (parens && x!=OPcat)
          fputc('(',fout);
        for (p=name;*p;p++)
        {
          if (*p=='_')
          {
            if (i==nb)
              die(n,"too few arguments for operator %s",name);
            printnodeparens(fout,arg[i++],1);
          }
          else
            fputc(*p,fout);
        }
        if (i!=nb)
          die(n,"too many arguments for operator %s",name);
        if (parens && x!=OPcat)
          fputc(')',fout);
      }
    }
    else if (isfunc(n,"_const_quote"))
      fprintf(fout,"'%s",value[tree[y].x].val.str);
    else if (isfunc(n,"_[_,]"))
    {
      printnode(fout,tree[y].x);
      fprintf(fout,"[");
      printnode(fout,tree[y].y);
      fprintf(fout,",]");
    }
    else
    {
      fprintf(fout,"%s",value[x].val.str);
      if (y>=0)
      {
        fprintf(fout,"(");
        if (y!=GNOARG)
          printnode(fout,y);
        fprintf(fout,")");
      }
    }
    break;
  case Frefarg:
    fprintf(fout,"&");
    printnode(fout,x);
    break;
  case Fvararg:
    printnode(fout,x);
    fprintf(fout,"[..]");
    break;
  case Fcall:
    printnode(fout,x);
    fprintf(fout,"(");
    printnode(fout,y);
    fprintf(fout,")");
    break;
  case Flambda:
    fprintf(fout,"(");
    printnode(fout,x);
    fprintf(fout,")->");
    printnode(fout,y);
    break;
  case Fdeffunc:
    fprintf(fout,"\n");
    printnode(fout,x);
    fprintf(fout,"=\n");
    genindent(fout);
    if (tree[y].f!=Fblock)
    {
      fprintf(fout,"{\n");
      indent++;
      genindent(fout);
    }
    printnode(fout,y);
    fprintf(fout,"\n");
    indent--;
    genindent(fout);
    if (tree[y].f!=Fblock)
    {
      fprintf(fout,"}\n");
      genindent(fout);
    }
    break;
  case Fblock:
    fprintf(fout,"{");
    fc=block+x;
    v=fc->ret;
    if (v>=0)
    {
      fprintf(fout,"/*=");
      printentry(fout,tree[v].x);
      fprintf(fout,"*/");
    }
    indent++;
    fprintf(fout,"\n");
    genindent(fout);
    if (fc->s.n)
    {
      int k=fc->s.n;
      fprintf(fout,"my(");
      for(i=0;i<k;i++)
      {
        if (i>0) fprintf(fout,", ");
        printentryname(fout,varstr(fc->c[i]));
      }
      fprintf(fout,");\n");
      genindent(fout);
    }
    printnode(fout,y);
    fprintf(fout,"\n");
    indent--;
    genindentline(fout,"}\n");
    genindent(fout);
    break;
  case Fnoarg:
    if ( n!= GNOARG )
      fprintf(fout,"gnil");
    break;
  case Ftag:
    printnodeparens(fout,x,1);
    fprintf(fout,":%s",GPname(y));
    break;
  }
}
void printnode(FILE *fout, int n)
{
  printnodeparens(fout,n,0);
}

void maketreeGRL_node(FILE *fout,int n)
{
  int x,y,f;
  if (n<2) return;
  x=tree[n].x;
  y=tree[n].y;
  f=tree[n].f;
  switch(tree[n].f)
  {
  case Ftag:
    fprintf(fout,"node: { title: \"%d\" label: \"(%s)\" }\n",n,GPname(y));
    fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color: red}\n",n,x);
    maketreeGRL_node(fout,x);
    break;
  default:
    if (f<FneedENTRY || f == Flambda)
    {
      fprintf(fout,"node: { title: \"%d\" label: \"%s\" }\n",n,Ffuncname[f]);
      if (x!=-1)
      {
        fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color:red}\n",n,x);
        maketreeGRL_node(fout,x);
        if (y!=-1 )
        {
          fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color:blue}\n",n,y);
          maketreeGRL_node(fout,y);
        }
      }
    }
    else
    {
      fprintf(fout,"node: { title: \"%d\" ",n);
      switch(f)
      {
      case Fsmall:
        fprintf(fout,"label: \"%d\" }\n",x);
        break;
      case Fdeffunc:
        fprintf(fout,"label: \"def(%s)\" }\n",entryname(x));
        break;
      case Fblock:
        fprintf(fout,"label: \"bloc(%d)\" }\n",x);
        break;
      default:
        fprintf(fout,"label: \"%s\" }\n",entryname(n));
      }
      if (tree[n].y!=-1)
      {
        fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color: blue}\n",n,y);
        maketreeGRL_node(fout,y);
      }
    }
  }
}

void maketreeGRL(FILE *fout,int n)
{
  fprintf(fout,"graph: { title:\"test\"\n");
  fprintf(fout,"xmax: 700 ymax: 700 x: 30 y: 30\n");
  fprintf(fout,"layout_downfactor: 8\n");
  fprintf(fout,"node: { title: \"%d\" label: \"GNIL\" }\n",GNIL);
  fprintf(fout,"node: { title: \"%d\" label: \"GNOARG\" }\n",GNOARG);
  maketreeGRL_node(fout,n);
  fprintf(fout,"}\n");

}

void maketree(FILE *fout,int n)
{
  switch(tree[n].f)
  {
  case Ftag:
    fprintf(fout,"(");
    maketree(fout,tree[n].x);
    fprintf(fout,",");
    fprintf(fout,"%s",GPname(tree[n].y));
    fprintf(fout,")");
    break;
  default:
    if (tree[n].f<FneedENTRY || tree[n].f==Flambda)
    {
      if (tree[n].x!=-1)
      {
        fprintf(fout,"(");
        maketree(fout,tree[n].x);
        if (tree[n].y!=-1 )
        {
          fprintf(fout,",");
          maketree(fout,tree[n].y);
        }
        fprintf(fout,")");
      }
    }
    else
    {
      if (tree[n].y!=-1)
      {
        fprintf(fout,"(");
        maketree(fout,tree[n].y);
        fprintf(fout,")");
      }
    }
  }
  fprintf(fout,"%s_%d",funcname(tree[n].f),n);
}
