/* 
 * Copyright (C) 2008 Bastien Nocera <hadess@hadess.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA.
 *
 */

/* To compile:
 * gcc -g -Wall -o lirc-generate-linux-input lirc-generate-linux-input.c `pkg-config --libs --cflags glib-2.0`
 */

/* lirc-generate-linux-input is used to generate the
 * lirc linux input layer configuration file from the
 * linux/input.h headers installed on the system.
 *
 * This program is not meant to be used by normal users,
 * just run every now and then by the gnome-lirc-properties
 * developers to update the headers compared to the Linux
 * headers */

#include <glib.h>
#include <string.h>
#include <linux/input.h>

#define START \
"# LIRC configuration file for receivers with Linux Input Layer driver.\n" \
"# generated by lirc-generate-linux-input\n" \
"#\n" \
"# brand: Generic\n" \
"# model: Linux Input Layer compatible Remote\n" \
"#\n\n" \
"begin remote\n" \
"\tname linux-input-layer\n" \
"\tbits 32\n" \
"\tbegin codes\n"

#define END \
"\tend codes\n" \
"end remote\n"

#define ADD 0x10000
#define SPACING_LEN 22

static gboolean
load_include (const char *path)
{
	GMappedFile *map;
	char *contents, **lines;
	guint i;

	map = g_mapped_file_new (path, FALSE, NULL);
	if (map == NULL) {
		g_message ("opening %s failed", path);
		return FALSE;
	}

	contents = g_strdup (g_mapped_file_get_contents (map));
	g_mapped_file_free (map);

	lines = g_strsplit (contents, "\n", -1);
	g_free (contents);

	g_print ("%s", START);

	for (i = 0; lines[i] != NULL; i++) {
		char **items;
		const char *line;
		double value;
		char *value_str;
		guint j, len;
		int code;

		line = lines[i] + strlen ("#define ");

		if (g_str_has_prefix (line, "KEY_") == FALSE &&
		    g_str_has_prefix (line, "BTN_") == FALSE)
			continue;
		if (g_str_has_prefix (line, "KEY_RESERVED") != FALSE)
			continue;
		if (g_str_has_prefix (line, "KEY_BRL_DOT1") != FALSE)
			break;

		items = g_strsplit (lines[i], "\t", 2);
		if (items[0] == NULL || items[1] == NULL) {
			g_strfreev (items);
			break;
		}

		value = g_strtod (items[1], NULL);
//		value_str = g_strdup_printf ("0x%x", (int) value + ADD);
		code = 0x80000000;
		code |= ((EV_KEY & 0x7fff) << 16);
		code |= (int) value;

		value_str = g_strdup_printf ("0x%x", (int) value + ADD);

		g_print ("\t\t%s", items[0] + strlen ("#define "));
		len = strlen (items[0] + strlen ("#define "));
		if (len >= SPACING_LEN) {
			g_warning ("increase SPACING_LEN to at least %d", len + 1);
			return FALSE;
		}
		for (j = 0; j < SPACING_LEN - len; j++)
			g_print (" ");
		g_print ("%s\n", value_str);
		g_free (value_str);

		g_strfreev (items);
	}

	g_strfreev (lines);

	g_print ("%s", END);

	return TRUE;
}

static void
set_warnings (void)
{
	GLogLevelFlags fatal_mask;

	fatal_mask = g_log_set_always_fatal (G_LOG_FATAL_MASK);
	fatal_mask |= G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL;
	g_log_set_always_fatal (fatal_mask);
}

int main (int argc, char **argv)
{
	if (argc != 1) {
		g_warning ("Usage: %s", argv[0]);
		return 1;
	}

	set_warnings ();

	if (load_include ("/usr/include/linux/input.h") == FALSE) {
		g_warning ("Failed to parse /usr/include/linux/input.h");
		return 1;
	}

	return 0;
}
