/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * GIMP Dimage color plugin
 * Copyright (C) 2002 Laurent HOUDARD <lhoudard@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <string.h>
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <libgimp/gimpmatrix.h>

#include "plugin-intl.h"
#include "transform.h"
#include "preview.h"
#include "dimage_color.h"

#define	PLUG_IN_NAME "plug_in_dimage_color"

DimageColorValues dc_vals = { TRUE, CURVE_DEFAULT };
DialogValues dc_int = { FALSE, TRUE, TRUE };

static void query (void);
static void run (gchar *name,
		 gint nparams,
		 GimpParam *param,
		 gint *nreturn_vals,
		 GimpParam **return_vals);

static GimpPDBStatusType dimage_color (GimpDrawable *drw);

static gint dimage_color_dialog (void);

GimpPlugInInfo PLUG_IN_INFO = {
  NULL,				/* init_proc  */
  NULL,				/* quit_proc  */
  query,			/* query_proc */
  run,				/* run_proc   */
};

MAIN ()
     
static void 
query (void)
{
  static GimpParamDef args[] = {
    { GIMP_PDB_INT32,    "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE,    "image",    "Input image" },
    { GIMP_PDB_DRAWABLE, "drawable", "Input drawable" },

    { GIMP_PDB_INT32,    "gamma", "Gamma + Tone correction" },
    { GIMP_PDB_INT32,    "tone_curve", "Tone curve" },
  };

  static gint nargs = sizeof (args) / sizeof (args[0]);
  
  gimp_plugin_domain_register (PLUGIN_NAME, LOCALEDIR);
  gimp_plugin_help_register (DATADIR "/help");

  gimp_install_procedure (PLUG_IN_NAME,
			  "Minolta DiMAGE Color to sRGB",
			  "Converts DiMAGE 5/7 pictures to sRGB colorspace",
			  "Laurent Houdard",
			  "Laurent Houdard",
			  "2002-02-05",
			  N_("<Image>/Image/Colors/Dimage Color to sRGB"),
			  "RGB*",
			  GIMP_PLUGIN,
			  nargs, 0, 
			  args, NULL);
}

static void
run (gchar *name,
     gint n_params,
     GimpParam *param,
     gint *nreturn_vals,
     GimpParam **return_vals)
{
  static GimpParam values[1];
  GimpPDBStatusType status = GIMP_PDB_SUCCESS;
  GimpDrawable *drawable;
  GimpRunModeType run_mode;
  gint32 image_id;
  gint32 drawable_id;

  *nreturn_vals = 1;
  *return_vals = values;

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;

  run_mode    = param[0].data.d_int32;
  image_id    = param[1].data.d_int32;
  drawable_id = param[2].data.d_int32;

  drawable = gimp_drawable_get (drawable_id);

  build_transform ();

  switch (run_mode)
    {
    case GIMP_RUN_INTERACTIVE:
      INIT_I18N_UI();
      gimp_get_data (PLUG_IN_NAME, &dc_vals);
      build_preview (drawable);
      if (! dimage_color_dialog ()) {
	return;
      }
      break;
      
    case GIMP_RUN_NONINTERACTIVE:
      INIT_I18N();
      dc_vals.gamma      = param[3].data.d_int32;
      dc_vals.tone_curve = param[4].data.d_int32;
      break;

    case GIMP_RUN_WITH_LAST_VALS:
      INIT_I18N();
      gimp_get_data (PLUG_IN_NAME, &dc_vals);
      break;

    default:
      break;
    }

  gimp_progress_init (_("Dimage Color to sRGB..."));
  gimp_tile_cache_ntiles (2 * (drawable->width / gimp_tile_width () + 1));

  status = dimage_color (drawable);

  if (run_mode != GIMP_RUN_NONINTERACTIVE)
    gimp_displays_flush ();

  if (run_mode == GIMP_RUN_INTERACTIVE && status == GIMP_PDB_SUCCESS)
    gimp_set_data (PLUG_IN_NAME, &dc_vals, sizeof(DimageColorValues));
  
  gimp_drawable_detach (drawable);
}

static void
dimage_color_ok_callback (GtkWidget *widget, gpointer data)
{
  dc_int.run = TRUE;
  gtk_widget_destroy (GTK_WIDGET (data));
}

static void
dimage_color_toggle_callback (GtkWidget *widget, gpointer data)
{
  gimp_toggle_button_update (widget, data);
  do_preview ();
}

static void
dimage_color_menu_item_update (GtkWidget *widget, gpointer data)
{
  gimp_menu_item_update (widget, data);
  if (dc_int.do_preview)
    do_preview ();
}

static gint
dimage_color_dialog (void)
{
  GtkWidget *dlg;
  GtkWidget *main_vbox;
  GtkWidget *preview_frame;
  GtkWidget *preview_vbox;
  GtkWidget *options_frame;
  GtkWidget *options_vbox;
  GtkWidget *visualize_out;
  GtkWidget *do_preview;
  GtkWidget *gamma;
  GtkWidget *tone_hbox;
  GtkWidget *tone_label;
  GtkWidget *tone_menu;

  gimp_ui_init ("dimage_color", TRUE);

  dlg = gimp_dialog_new(_("Dimage color"), "dimage_color",
			gimp_standard_help_func, "filters/dimage_color.html",
			GTK_WIN_POS_MOUSE,
			FALSE, TRUE, FALSE,

			_("OK"), dimage_color_ok_callback,
			NULL, NULL, NULL, TRUE, FALSE,
			_("Cancel"), gtk_widget_destroy,
			NULL, 1, NULL, FALSE, TRUE,

			NULL);

  gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit),
		      NULL);

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (main_vbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), main_vbox, 
		      TRUE, TRUE, 0);

  /* Preview */
  preview_frame = gtk_frame_new (_("Preview"));
  gtk_widget_show (preview_frame);
  gtk_box_pack_start (GTK_BOX (main_vbox), preview_frame, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (preview_frame), 6);

  preview_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (preview_vbox);
  gtk_container_add (GTK_CONTAINER (preview_frame), preview_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (preview_vbox), 6);

  preview_new (preview_vbox);

  do_preview = gtk_check_button_new_with_label (_("Do Preview"));
  gtk_widget_show (do_preview);
  gtk_box_pack_start (GTK_BOX (preview_vbox), do_preview, FALSE, FALSE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (do_preview), 
				dc_int.do_preview);
  gtk_signal_connect (GTK_OBJECT (do_preview), "toggled",
                      GTK_SIGNAL_FUNC (dimage_color_toggle_callback),
                      &dc_int.do_preview);

  visualize_out = 
    gtk_check_button_new_with_label (_("Visualize out of gamut points"));
  gtk_widget_show (visualize_out);
  gtk_box_pack_start (GTK_BOX (preview_vbox), visualize_out, FALSE, FALSE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (visualize_out), 
				dc_int.visualize_out);
  gtk_signal_connect (GTK_OBJECT (visualize_out), "toggled",
                      GTK_SIGNAL_FUNC (dimage_color_toggle_callback),
                      &dc_int.visualize_out);

  /* Parameter settings */
  options_frame = gtk_frame_new (_("Parameter setting"));
  gtk_widget_show (options_frame);
  gtk_box_pack_start (GTK_BOX (main_vbox), options_frame, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (options_frame), 6);

  options_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (options_vbox);
  gtk_container_add (GTK_CONTAINER (options_frame), options_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (options_vbox), 6);

  gamma = gtk_check_button_new_with_label (_("Gamma and tone correction"));
  gtk_widget_show (gamma);
  gtk_box_pack_start (GTK_BOX (options_vbox), gamma, FALSE, FALSE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gamma), 
				dc_vals.gamma);
  gtk_signal_connect (GTK_OBJECT (gamma), "toggled",
                      GTK_SIGNAL_FUNC (dimage_color_toggle_callback),
                      &dc_vals.gamma);

  tone_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (tone_hbox);
  gtk_box_pack_start (GTK_BOX (options_vbox), tone_hbox, FALSE, FALSE, 0);

  tone_label = gtk_label_new (_("Tone curve"));
  gtk_widget_show (tone_label);
  gtk_box_pack_start (GTK_BOX (tone_hbox), tone_label, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (tone_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_padding (GTK_MISC (tone_label), 20, 0);

  tone_menu = 
    gimp_option_menu_new2 (FALSE, dimage_color_menu_item_update,
			   &dc_vals.tone_curve, (gpointer) dc_vals.tone_curve,

			   _("Default"), (gpointer) CURVE_DEFAULT, NULL,
			   _("Flat"),    (gpointer) CURVE_FLAT,    NULL,

			   /* Add custom curves here */
				     
			   NULL);

  gtk_widget_show (tone_menu);
  gtk_box_pack_start (GTK_BOX (tone_hbox), tone_menu, FALSE, FALSE, 0);

  gtk_widget_show (dlg);

  gtk_main ();
  gdk_flush ();

  return dc_int.run;
}

static GimpPDBStatusType
dimage_color (GimpDrawable *drawable)
{
  GimpPixelRgn src_rgn, dest_rgn;
  guchar *src, *s;
  guchar *dest, *d;
  gint    progress, max_progress;
  gint    x1, y1, x2, y2;
  gint    x, y;
  gpointer pr;

  /* Get selection area */
  gimp_drawable_mask_bounds (drawable->id, &x1, &y1, &x2, &y2);

  progress = 0;
  max_progress = (x2 - x1) * (y2 - y1);
  
  gimp_pixel_rgn_init (&src_rgn, drawable,
		       x1, y1, (x2 - x1), (y2 - y1), FALSE, FALSE);
  gimp_pixel_rgn_init (&dest_rgn, drawable,
		       x1, y1, (x2 - x1), (y2 - y1), TRUE, TRUE);
  
  for (pr = gimp_pixel_rgns_register (2, &src_rgn, &dest_rgn);
       pr != NULL;
       pr = gimp_pixel_rgns_process (pr))
    {
      src = src_rgn.data;
      dest = dest_rgn.data;
      
      for (y = 0; y < src_rgn.h; y++)
	{
	  s = src;
	  d = dest;

	  for (x = 0; x < src_rgn.w; x++)
	    {
	      transform_point (s, d, FALSE);
	      s += src_rgn.bpp;
	      d += dest_rgn.bpp;
	    }
	  
	  src += src_rgn.rowstride;
	  dest += dest_rgn.rowstride;
	}

      /* Update progress */
      progress += src_rgn.w * src_rgn.h;
      gimp_progress_update ((gdouble) progress / (gdouble) max_progress);
    }

  /* Update the region */
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->id, TRUE);
  gimp_drawable_update (drawable->id, x1, y1, (x2 - x1), (y2 - y1));

  return GIMP_PDB_SUCCESS;
}
