# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

import os

from gi.repository import (
    GdkPixbuf,
    Gtk,
    GtkSource,
    )

from gdp import (
    Config,
    config,
    )
from gdp import complete
from gdp.complete import (
    BaseGenerator,
    DynamicProposal,
    DynamicProvider,
    MarkupGenerator,
    PangoDoc,
    PythonProposal,
    PythonGenerator,
    Completer,
    TextGenerator,
    )
from testing import SignalTester
from testing import GeditTestCase


class BaseGeneratorTestCase(GeditTestCase):

    def test_properies(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        text = document.get_text(
            document.get_start_iter(), document.get_end_iter(), True)
        generator = BaseGenerator(document)
        self.assertIn('plugins/gdp/data/snark12.txt', generator.file_path)
        self.assertEqual(text, generator.text)
        self.assertEqual(r'[\w_]', generator.word_char.pattern)
        self.assertEqual('', generator.string_before_cursor)

    def test_get_words_error(self):
        # Subclass must implement get_words.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        generator = BaseGenerator(document)
        self.assertRaises(NotImplementedError, generator.get_words, 'Ba')


class TextGeneratorTestCase(GeditTestCase):

    def test_get_words_without_prefix(self):
        # Subclass must implement get_words.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        generator = TextGenerator(document)
        is_authoritative, proposals = generator.get_words()
        self.assertIs(False, is_authoritative)
        self.assertEqual(1987, len(proposals))
        self.assertIsInstance(proposals[0], DynamicProposal)
        self.assertEqual(
            ['000', '000x100', '03', '08', '1', '10', '1992', '2', '20'],
            [proposal._word for proposal in sorted(proposals)][0:9])

    def test_get_words_with_prefix(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        generator = TextGenerator(document, prefix='Ba')
        is_authoritative, proposals = generator.get_words()
        self.assertEqual(16, len(proposals))
        self.assertEqual(
            ['BAKER', 'BANKER', 'BARRISTER', 'Baker', 'Bandersnatch'],
            [proposal._word for proposal in sorted(proposals)][0:5])

    def test_get_words_case_insensitivity(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        generator = TextGenerator(document)
        is_authoritative, lowercase_proposals = generator.get_words('snar')
        is_authoritative, uppercase_proposals = generator.get_words('SNAR')
        self.assertEqual(
            ['SNARK', 'Snark', 'Snarks', 'snark12', 'snarked'],
            [proposal._word for proposal in sorted(lowercase_proposals)])
        self.assertEqual(lowercase_proposals, uppercase_proposals)

    def test_get_words_with_hyphenation(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        generator = TextGenerator(document)
        is_authoritative, proposals = generator.get_words('What-')
        self.assertEqual(2, len(proposals))
        self.assertEqual(
            ['What-was-his-name', 'What-you-may-call-um'],
            [proposal._word for proposal in sorted(proposals)])

    def test_get_words_with_undescored(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/hackerese.txt')
        generator = TextGenerator(document)
        is_authoritative, proposals = generator.get_words('_')
        self.assertEqual(6, len(proposals))
        self.assertEqual(
            ['_', '__call__', '__dict__', '__init__', '_dict', '_private'],
            [proposal._word for proposal in sorted(proposals)])


class MarkupGeneratorTestCase(GeditTestCase):

    def test_string_before_cursor(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        generator = MarkupGenerator(document)
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_lines(2)
        cursor_iter.forward_chars(10)
        document.place_cursor(cursor_iter)
        self.assertEqual(r'[^<>]', generator.word_char.pattern)
        self.assertEqual('title', generator.string_before_cursor)

    def test_get_words_outside_tag(self):
        # There are no proposals when the cursor is outside of a tag.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        document.place_cursor(document.get_start_iter())
        generator = MarkupGenerator(document)
        is_authoritative, proposals = generator.get_words()
        self.assertIs(MarkupGenerator.OUTSIDE, generator.get_cursor_context())
        self.assertEqual(set(), proposals)
        self.assertIs(False, is_authoritative)

    def test_get_words_inside_open_tag(self):
        # The proposals match the unique tags uses in the document.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        generator = MarkupGenerator(document)
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_lines(2)
        cursor_iter.forward_chars(10)
        document.place_cursor(cursor_iter)
        is_authoritative, proposals = generator.get_words('')
        self.assertIs(
            MarkupGenerator.INSIDE_OPEN, generator.get_cursor_context())
        self.assertIs(True, is_authoritative)
        self.assertEqual('title', generator.string_before_cursor)
        self.assertEqual(
            ['body', 'br', 'div', 'em', 'h1', 'h2', 'head', 'html',
             'p', 'pre', 'title'],
            [proposal._word for proposal in sorted(proposals)])

    def test_get_words_inside_attributes_tag(self):
        # The proposals match the unique attributes used in the document
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        generator = MarkupGenerator(document)
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_lines(2)
        cursor_iter.forward_chars(10)
        document.place_cursor(cursor_iter)
        document.insert_at_cursor(" c")
        is_authoritative, proposals = generator.get_words()
        self.assertIs(
            MarkupGenerator.INSIDE_ATTRIBUTES, generator.get_cursor_context())
        self.assertIs(True, is_authoritative)
        self.assertEqual('title c', generator.string_before_cursor)
        self.assertEqual(
            ['class'], [proposal._word for proposal in sorted(proposals)])

    def test_get_words_inside_close_tag(self):
        # The proposals match the unique tags that are open before the cursor.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        generator = MarkupGenerator(document)
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_lines(2)
        cursor_iter.forward_chars(37)
        document.place_cursor(cursor_iter)
        is_authoritative, proposals = generator.get_words('')
        self.assertIs(
            MarkupGenerator.INSIDE_CLOSE, generator.get_cursor_context())
        self.assertIs(True, is_authoritative)
        self.assertEqual('/', generator.string_before_cursor)
        self.assertEqual(
            ['head>', 'html>', 'title>'],
            [proposal._word for proposal in sorted(proposals)])


class PythonGeneratorTestCase(GeditTestCase):

    def test_kwlist(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        generator = PythonGenerator(document)
        proposals = set(proposal._word for proposal in generator.kwlist)
        self.assertTrue(set(['and', 'as', 'assert']).issubset(proposals))

    def test_builtin(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        generator = PythonGenerator(document)
        proposals = set(proposal._word for proposal in generator.builtin)
        self.assertTrue(set(['list', 'str', 'xrange']).issubset(proposals))

    def test_get_local_symbols(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        self.assertIs(None, generator._local_symsbols)
        generator.get_words()
        self.assertIsNone(None, generator._local_symsbols)
        self.assertIs(
            generator._local_symsbols, generator.get_local_symbols())
        symbols = [pp._word for pp in generator.get_local_symbols()[0:3]]
        self.assertEqual(['type', '__metaclass__', '__all__'], symbols)

    def test_get_words_returns_pythonproposal(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        generator = PythonGenerator(document)
        document.place_cursor(document.get_end_iter())
        document.insert_at_cursor('os.path.j')
        is_authoritative, proposals = generator.get_words('j')
        proposal = proposals[0]
        self.assertIsInstance(proposal, PythonProposal)

    def test_pythonproposal_do_get_info(self):
        # do_get_info returns pydoc for the identifier.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        generator = PythonGenerator(document)
        document.place_cursor(document.get_end_iter())
        document.insert_at_cursor('os.path.j')
        is_authoritative, proposals = generator.get_words('j')
        proposal = proposals[0]
        text = proposal.do_get_info()
        self.assertIs(os.path.join, proposal._info)
        self.assertIn(
            '<b>join</b>(a, *p)\n    Join two or more pathname', text)

    def test_pythonproposal_do_get_info_none(self):
        # do_get_info returns None if the info is None.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words('A')
        proposal = proposals[0]
        proposal._info = None
        self.assertIs('', proposal.do_get_info())

    def test_get_words_with_prefix(self):
        # Python identifiers are matched, not words.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words('A')
        self.assertIn('An ',  generator.text)
        self.assertEqual(3, len(proposals))
        self.assertEqual(
            ['ArithmeticError', 'AssertionError', 'AttributeError'],
            [proposal._word for proposal in sorted(proposals)])

    def test_get_words_with_keywords(self):
        # Python keywords are matches.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words('wh')
        self.assertEqual(1, len(proposals))
        self.assertEqual(
            ['while'],
            [proposal._word for proposal in sorted(proposals)])

    def test_get_words_gtk_module(self):
        python_file = self.make_file('\n'.join([
            'from gi.repository import Gtk',
            'justification = Gtk.Justification.',
            ]))
        window, view, document = self.make_gedit(python_file.name)
        document.place_cursor(document.get_end_iter())
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words()
        self.assertEqual(
            ['CENTER', 'FILL', 'LEFT', 'RIGHT'],
            [proposal._word for proposal in sorted(proposals)][0:4])

    def test_get_words_python_module(self):
        python_file = self.make_file('\n'.join([
            'import os.path',
            'dirname = os.path.',
            ]))
        window, view, document = self.make_gedit(python_file.name)
        document.place_cursor(document.get_end_iter())
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words('a')
        self.assertEqual(
            ['abspath', 'altsep'],
            [proposal._word for proposal in sorted(proposals)])

    def test_get_words_is_authoratative(self):
        # The matches are authoritative when a namespace is
        # matched without a prefix.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.set_line(document.get_line_count() - 6)
        generator = PythonGenerator(document)
        is_authoritative, proposals = generator.get_words('wh')
        self.assertIs(False, is_authoritative)
        document.place_cursor(document.get_end_iter())
        document.insert_at_cursor('os.path.j')
        is_authoritative, proposals = generator.get_words('j')
        self.assertIs(True, is_authoritative)

    def test_get_parsable_text_at_start_of_block(self):
        # If the line being edited starts a block, get_parsable_text replaces
        # line with code (if True:) that will guarantee the module will
        # compile.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.set_line(76)
        document.place_cursor(cursor_iter)
        document.insert_at_cursor(
            '    found = False\n'
            '    while not found:\n'
            '        found = True\n')
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.backward_lines(2)
        document.place_cursor(cursor_iter)
        parsable_lines = generator._get_parsable_text().splitlines()
        self.assertEqual('    found = False', parsable_lines[76])
        self.assertEqual('    if True:', parsable_lines[77])
        self.assertEqual('        found = True', parsable_lines[78])

    def test_get_parsable_text_block_indentation(self):
        # When block line has the same indentation as the next line,
        # or it is the last, it is replaced with 'pass' to compile.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        document.place_cursor(document.get_start_iter())
        generator = PythonGenerator(document)
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.set_line(76)
        document.place_cursor(cursor_iter)
        document.insert_at_cursor(
            '    found = False\n'
            '    while not found:\n'
            '        found = True\n')
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.backward_lines(1)
        document.place_cursor(cursor_iter)
        parsable_lines = generator._get_parsable_text().splitlines()
        self.assertEqual('    found = False', parsable_lines[76])
        self.assertEqual('    while not found:', parsable_lines[77])
        self.assertEqual('        pass', parsable_lines[78])


def handler(proposal, piter):
    return proposal.get_text()


def create_context(view, user_requested=True):
    document = view.get_buffer()
    iter_ = document.get_iter_at_mark(document.get_insert())
    completion = view.get_completion()
    if user_requested:
        activation = GtkSource.CompletionActivation.USER_REQUESTED
    else:
        activation = GtkSource.CompletionActivation.INTERACTIVE
    return GtkSource.CompletionContext(
       iter=iter_, completion=completion, activation=activation)


class DynamicProposalTestCase(GeditTestCase):

    def test_init(self):
        proposal = DynamicProposal('word', 'info')
        self.assertEqual('word', proposal._word)
        self.assertEqual('info', proposal._info)

    def test_equal(self):
        proposal_1 = DynamicProposal('word', 'info')
        proposal_2 = DynamicProposal('word', 'info')
        self.assertTrue(proposal_1 == proposal_2)

    def test_lt(self):
        proposal_1 = DynamicProposal('word', 'info')
        proposal_2 = DynamicProposal('word5', 'info')
        self.assertTrue(proposal_1 < proposal_2)

    def test_hash(self):
        proposal_1 = DynamicProposal('word', 'info')
        proposal_2 = DynamicProposal('word', 'info')
        proposal_3 = DynamicProposal('other', 'info')
        self.assertEqual(proposal_1.__hash__(), proposal_2.__hash__())
        self.assertNotEqual(proposal_1.__hash__(), proposal_3.__hash__())

    def test_do_changed(self):
        proposal = DynamicProposal('word', 'info')
        self.assertFalse(proposal.do_changed())

    def test_do_equal(self):
        self.test_equal()

    def test_do_hash(self):
        self.test_hash()

    def test_do_get_text(self):
        proposal = DynamicProposal('word', 'info')
        self.assertEqual('word', proposal.do_get_text())

    def test_do_get_label(self):
        proposal = DynamicProposal('word', 'info')
        self.assertEqual('word', proposal.do_get_label())

    def test_do_get_markup(self):
        proposal = DynamicProposal('<b>word</b>', 'info')
        self.assertEqual('&lt;b&gt;word&lt;/b&gt;', proposal.do_get_markup())

    def test_do_get_info(self):
        proposal = DynamicProposal('word', 'info')
        self.assertEqual('info', proposal.do_get_info())

    def test_do_get_icon(self):
        proposal = DynamicProposal('word', 'info')
        self.assertIs(None, proposal.do_get_icon())


class PythonProposalTestCase(GeditTestCase):

    def test_do_get_info(self):
        proposal = PythonProposal('sorted', sorted)
        info = PangoDoc().document(sorted)
        self.assertEqual(info, proposal.do_get_info())
        self.assertIn('<b>sorted</b>(...)\n', proposal.do_get_info())

    def test_do_get_info_none(self):
        proposal = PythonProposal('fnord', None)
        self.assertEqual('', proposal._info)
        self.assertEqual('', proposal.do_get_info())


class DynamicProviderTestCase(GeditTestCase):

    def test_init(self):
        provider = DynamicProvider('tester', handler)
        self.assertEqual('tester', provider._name)
        self.assertEqual(handler, provider.handler)
        self.assertIs(None, provider.info_widget)
        self.assertIs(None, provider.mark)
        self.assertIsInstance(provider.icon, GdkPixbuf.Pixbuf)

    def test_do_get_activation_auto_false(self):
        provider = DynamicProvider('tester', handler)
        original_value = config.getboolean('completer', 'suggest_completions')
        self.addCleanup(
            config.set, 'completer', 'suggest_completions',
            str(original_value))
        config.set('completer', 'suggest_completions', str(False))
        self.assertEqual(
            GtkSource.CompletionActivation.USER_REQUESTED,
            provider.do_get_activation())

    def test_do_get_activation_auto_true(self):
        provider = DynamicProvider('tester', handler)
        original_value = config.getboolean('completer', 'suggest_completions')
        self.addCleanup(
            config.set, 'completer', 'suggest_completions',
            str(original_value))
        config.set('completer', 'suggest_completions', str(True))
        self.assertEqual(
            (GtkSource.CompletionActivation.USER_REQUESTED |
             GtkSource.CompletionActivation.INTERACTIVE),
            provider.do_get_activation())

    def test_do_match_requested(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        context = create_context(view)
        provider = DynamicProvider('tester', handler)
        self.assertIs(True, provider.do_match(context))

    def test_do_match_interactive_short_prefix(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        context = create_context(view, False)
        provider = DynamicProvider('tester', handler)
        self.assertIs(False, provider.do_match(context))

    def test_do_match_interactive_long_prefix(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_cursor_positions(47)
        document.place_cursor(cursor_iter)
        context = create_context(view, False)
        provider = DynamicProvider('tester', handler)
        self.assertEqual('Hunt', provider.get_word(context))
        self.assertIs(True, provider.do_match(context))

    def test_get_proposals(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        context = create_context(view)
        provider = DynamicProvider('tester', handler)
        proposals = provider.get_proposals(prefix='', context=context)
        self.assertEqual(1987, len(proposals))
        self.assertIsInstance(proposals[0], DynamicProposal)
        self.assertEqual(
            ['000', '000x100', '03', '08', '1', '10', '1992', '2', '20'],
            [proposal._word for proposal in sorted(proposals)][0:9])

    def test_do_populate_interactive_one_match(self):
        # There is is only one match in interactive proposals is an
        # empty list because the match is the text at the cursor.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_cursor_positions(76)
        document.place_cursor(cursor_iter)
        context = create_context(view, False)
        provider = DynamicProvider('tester', handler)
        proposals = provider._do_populate(context=context)
        self.assertEqual('29th', provider.get_word(context))
        self.assertIs(0, len(proposals))

    def test_get_generator_text(self):
        # The text generator is never returned because get_proposals will
        # use it in non-authoritative cases.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        provider = DynamicProvider('tester', handler)
        generator = provider.get_generator(document, '')
        self.assertIs(None, generator)

    def test_get_generator_python(self):
        # The PythonGenerator is used for Python files.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        provider = DynamicProvider('tester', handler)
        generator = provider.get_generator(document, '')
        self.assertIsInstance(generator, PythonGenerator)

    def test_get_generator_python_cache(self):
        # The PythonGenerator is reused when the lines has not moved.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.py')
        cursor_iter = document.get_start_iter()
        cursor_iter.forward_lines(14)
        document.place_cursor(cursor_iter)
        document.insert_at_cursor("impor")
        provider = DynamicProvider('tester', handler)
        self.assertEqual(0, provider.last_line_no)
        self.assertIs(None, provider.last_python_generator)
        generator = provider.get_generator(document, 'impor')
        self.assertIsNone(None, provider.last_python_generator)
        self.assertIs(provider.last_python_generator, generator)
        self.assertEqual(14, provider.last_line_no)
        self.assertIs(generator, provider.get_generator(document, ''))
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_lines(1)
        document.place_cursor(cursor_iter)
        self.assertIsNot(generator, provider.get_generator(document, ''))

    def test_get_generator_markup(self):
        # The MarkupGenerator is use for XML-like files.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        provider = DynamicProvider('tester', handler)
        generator = provider.get_generator(document, '')
        self.assertIsInstance(generator, MarkupGenerator)

    def test_do_get_info_widget(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark.html')
        provider = DynamicProvider('tester', handler)
        widget = provider.do_get_info_widget(None)
        self.assertIsInstance(widget, Gtk.ScrolledWindow)


class CompleterTestCase(GeditTestCase):

    def assertPrefixEquals(self, expected, prefix):
        result = (prefix[0], prefix[1].get_offset(), prefix[2].get_offset())
        self.assertEqual(expected, result)

    def test_init(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        self.assertIs(window, controller.window)

    def test_set_view_with_none(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        controller.set_view(None)
        self.assertIs(None, controller.view)
        self.assertIs(None, controller.completion)
        self.assertIs(None, controller.provider)
        self.assertEqual({}, controller.signal_ids)

    def test_set_view_with_view(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        controller.set_view(None)
        controller.set_view(view)
        self.assertIs(view, controller.view)
        self.assertIs(view.get_completion(), controller.completion)
        self.assertIsInstance(controller.provider, DynamicProvider)
        self.assertEqual(
            controller.on_proposal_activated, controller.provider.handler)
        self.assertEqual(
            ['destroy', 'notify::editable'],
            sorted(controller.signal_ids.keys()))

    def test_show_completion(self):
        # show_completion emits the show-completion signal on the view.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        signal_tester = SignalTester(['view'])
        view.connect('show-completion', signal_tester.receiver)
        controller.show_completion()
        self.assertIs(view, signal_tester.view)

    def test_get_word_prefix(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        self.assertEqual('This is the Pro', controller.text[0:15])
        self.assertPrefixEquals(
            (None, 0, 0), controller.get_word_prefix(document))
        cursor_iter.forward_cursor_positions(6)
        document.place_cursor(cursor_iter)
        self.assertPrefixEquals(
            ('i', 5, 6), controller.get_word_prefix(document))
        cursor_iter.forward_cursor_positions(1)
        document.place_cursor(cursor_iter)
        self.assertPrefixEquals(
            ('is', 5, 7), controller.get_word_prefix(document))

    def test_get_word_prefix_word_boundry(self):
        # Spaces and punctuation are not part of the word prefix.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_cursor_positions(5)
        document.place_cursor(cursor_iter)
        self.assertEqual('This is the Pro', controller.text[0:15])
        self.assertPrefixEquals(
            (None, 5, 5), controller.get_word_prefix(document))
        cursor_iter.forward_cursor_positions(58)
        document.place_cursor(cursor_iter)
        self.assertEqual(' of the Snark.\n *Thi', controller.text[50: 70])
        self.assertPrefixEquals(
            ('Snark', 58, 63), controller.get_word_prefix(document))
        cursor_iter.forward_cursor_positions(3)
        document.place_cursor(cursor_iter)
        self.assertPrefixEquals(
            (None, 66, 66), controller.get_word_prefix(document))

    def test_get_word_prefix_underscore(self):
        # Underscores are part of the word prefix.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/hackerese.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_cursor_positions(464)
        document.place_cursor(cursor_iter)
        self.assertEqual(' syntax_completer\n    _', controller.text[447:470])
        self.assertPrefixEquals(
            ('syntax_completer', 448, 464),
            controller.get_word_prefix(document))

    def test_get_insert_word_insert(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_cursor_positions(12)
        document.place_cursor(cursor_iter)
        self.assertEqual(
            'This is the Project Gutenberg Etext of The Hunting of the Snark',
            controller.text[0:63])
        controller.insert_word('Est ', cursor_iter)
        self.assertEqual(
            'This is the Est Project Gutenberg Etext of The Hunting of the S',
            controller.text[0:63])

    def test_get_insert_word_replace(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_cursor_positions(11)
        document.place_cursor(cursor_iter)
        start_iter = document.get_start_iter()
        start_iter.forward_cursor_positions(8)
        self.assertEqual(
            'This is the Project Gutenberg Etext of The Hunting of the Snark',
            controller.text[0:63])
        controller.insert_word('a', start_iter)
        self.assertEqual(
            'This is a Project Gutenberg Etext of The Hunting of the Snark',
            controller.text[0:61])

    def test_on_proposal_activated_with_word(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        document.place_cursor(document.get_start_iter())
        cursor_iter = document.get_iter_at_mark(document.get_insert())
        cursor_iter.forward_cursor_positions(12)
        document.place_cursor(cursor_iter)
        self.assertEqual(
            'This is the Project Gutenberg Etext of The Hunting of the Snark',
            controller.text[0:63])
        proposal = DynamicProposal('esteemd')
        success = controller.on_proposal_activated(proposal, None)
        self.assertIs(True, success)
        self.assertEqual(
            'This is the esteemdProject Gutenberg Etext of The Hunting of th',
            controller.text[0:63])

    def test_on_suggest_completions_toggled(self):
        config_file = self.make_config(complete, Config)
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        menu_item = Gtk.CheckMenuItem(label='test')
        menu_item.set_active(True)
        controller.on_suggest_completions_toggled(menu_item)
        self.assertIs(
            True, complete.config.getboolean(
                'completer', 'suggest_completions'))
        self.assertIn('suggest_completions = True', config_file.read())
        menu_item.set_active(False)
        controller.on_suggest_completions_toggled(menu_item)
        self.assertIs(
            False, complete.config.getboolean(
                'completer', 'suggest_completions'))

    def test_deactivate_on_view_destroy(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        controller = Completer(window)
        self.assertIn('destroy', controller.signal_ids.keys())
        self.assertIs(view, controller.view)
        view.destroy()
        self.assertIs(None, controller.view)
