import desktop
import gtk


#
# Class for transparent windows.
#
class GlassWindow(gtk.Window):

    # transparency modes
    TRANSPARENCY_NONE = 0
    TRANSPARENCY_SCREENSHOT = 1
    TRANSPARENCY_PSEUDO = 2

    

    #
    # Constructor.
    #
    def __init__(self, wintype = gtk.WINDOW_TOPLEVEL):

        # the transparency mode to use
        self.__transparency = self.TRANSPARENCY_PSEUDO

        # transparency level of the window (0..255)
        self.__bg_alpha = 255
        # the background image of the window
        self.__bg_image = None
        # the desktop background
        self.__bg_root = None

        # remember the current window position to detect movements
        self.__position = (0, 0)


        gtk.Window.__init__(self, wintype)
        self.connect("configure-event", self.__on_configure)



    #
    # Observer method for the background.
    # Connect this method to the BG watcher.
    #
    def bg_observer(self, src, cmd):

        self.update_bg()
        


    #
    # Sets the transparency mode of this window.
    #
    def set_transparency_mode(self, mode):

        self.__transparency = mode



    #
    # Sets the background image and the transparency level of the window.
    #
    def set_bg_image(self, filename, alpha):

        self.__bg_alpha = alpha

        # load image
        if (filename):
            # this is an ugly hack for avoiding memory leaks in pygtk
            img = gtk.Image()
            img.set_from_file(filename)
            self.__bg_image = img.get_pixbuf()
            
            # this would cause memory leaks in pygtk versions < 1.99.14
            #self.__bg_image = gtk.gdk.pixbuf_new_from_file(filename)
                
        # if no image is available, create a solid background
        if (not filename or not self.__bg_image):
            self.__bg_image = gtk.gdk.Pixbuf(0, 1, 8, 10, 10)
            self.__bg_image.fill(0xffffff)

        gtk.idle_add(self.update_bg)




    #
    # Displays the background image.
    #
    def __display_bg(self):

        if (not self.__bg_root): return

        width = self.__bg_root.get_width()
        height = self.__bg_root.get_height()

        # we cannot use Pixbuf.copy() here because of a memory leak
        # bug in pygtk versions < 1.99.14
        image = gtk.gdk.Pixbuf(0, 1, 8, width, height)
        self.__bg_root.copy_area(0, 0, width, height, image, 0, 0)

        # let GdkPixbuf do the alpha-blending
        self.__bg_image.composite(image, 0, 0, width, height,
                                  0, 0, 1, 1, gtk.gdk.INTERP_BILINEAR,
                                  self.__bg_alpha)


        # set as window bg
        pix, mask = image.render_pixmap_and_mask()

        style = gtk.Style()
        style.bg_pixmap[gtk.STATE_NORMAL] = pix
        self.set_style(style)

        # tell gtk to redraw this window ASAP
        self.queue_draw()




    #
    # Updates the background for transparency.
    #
    def update_bg(self, noscreenshot = 0):

        if (not self.window): return
        
        x, y = self.window.get_origin()
        width, height = self.window.get_size()
        if (self.__bg_image and width > 10 and height > 10):
            self.__capture_bg(x, y, width, height, noscreenshot)
            self.__display_bg()



    #
    # Captures the background to create transparency.
    #
    def __capture_bg(self, x, y, width, height, noscreenshot = 0):

        # opaque background
        if (self.__transparency == self.TRANSPARENCY_NONE):
            self.__bg_root = gtk.gdk.Pixbuf(0, 1, 8, width, height)
            # TODO: let the user set the bg color
            self.__bg_root.fill(0xddddee)
            return
            
        # try capturing the background
        elif (self.__transparency == self.TRANSPARENCY_PSEUDO):
            try:
                self.__bg_root = desktop.get_wallpaper(x, y, width, height)
                return
            except:
                try:
                    self.__bg_root = desktop.get_wallcolor(width, height)
                    return
                except:
                    self.__bg_root = gtk.gdk.Pixbuf(0, 1, 8, width, height)
                    self.__bg_root.fill(0xddddee)
                    return
                #end try
            #end try
        #end if

        
        # use screenshot transparency if capturing has failed or transparency
        # mode is set to screenshot transparency
        if (noscreenshot): return  # must not make screenshot now

        self.hide()
        self.move(-1000, 0)
        self.queue_draw()

        # wait long enough for the window to disappear; does this work always?
        while (gtk.gdk.events_pending()): gtk.mainiteration()
        gtk.mainiteration()
        
        self.__bg_root = desktop.get_wallpaper_fallback(x, y, width, height)
        self.show()
        self.move(x, y)



    #
    # Reacts on moving the window.
    #
    def __on_configure(self, src, event):

        pos = self.window.get_origin()
        if (pos != self.__position):
            self.__position = pos

            # don't make screenshots while moving the window
            self.update_bg(noscreenshot = 1)
