/*
 * Copyright (c) 2021-2022 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#import "ShaderStage.h"
#import <wtf/EnumeratedArray.h>
#import <wtf/FastMalloc.h>
#import <wtf/HashMap.h>
#import <wtf/HashTraits.h>
#import <wtf/Ref.h>
#import <wtf/RefCountedAndCanMakeWeakPtr.h>
#import <wtf/TZoneMalloc.h>
#import <wtf/Vector.h>
#import <wtf/WeakPtr.h>

struct WGPUPipelineLayoutImpl {
};

namespace WebGPU {

class BindGroup;
class BindGroupLayout;
class Device;

// https://gpuweb.github.io/gpuweb/#gpupipelinelayout
class PipelineLayout : public RefCountedAndCanMakeWeakPtr<PipelineLayout>, public WGPUPipelineLayoutImpl {
    WTF_MAKE_TZONE_ALLOCATED(PipelineLayout);
public:
    static Ref<PipelineLayout> create(std::optional<Vector<Ref<BindGroupLayout>>>&& bindGroupLayouts, bool isAutogenerated, uint64_t uniqueId, Device& device)
    {
        return adoptRef(*new PipelineLayout(WTFMove(bindGroupLayouts), isAutogenerated, uniqueId, device));
    }
    static Ref<PipelineLayout> createInvalid(Device& device)
    {
        return adoptRef(*new PipelineLayout(device));
    }

    ~PipelineLayout();

    void setLabel(String&&);

    bool isValid() const { return m_isValid; }

    bool operator==(const PipelineLayout&) const;

    bool isAutoLayout() const;
    size_t numberOfBindGroupLayouts() const { return m_bindGroupLayouts ? m_bindGroupLayouts->size() : 0; }
    BindGroupLayout& bindGroupLayout(size_t) const;
    BindGroupLayout* optionalBindGroupLayout(size_t) const;
    RefPtr<BindGroupLayout> protectedOptionalBindGroupLayout(size_t) const;

    Device& device() const { return m_device; }
    void makeInvalid();
    size_t sizeOfVertexDynamicOffsets() const;
    size_t sizeOfFragmentDynamicOffsets() const;
    size_t sizeOfComputeDynamicOffsets() const;

    bool updateVertexOffsets(uint32_t, const Vector<uint32_t>&, std::span<uint32_t> destination);
    bool updateFragmentOffsets(uint32_t, const Vector<uint32_t>&, std::span<uint32_t> destination);
    bool updateComputeOffsets(uint32_t, const Vector<uint32_t>&, std::span<uint32_t> destination);
    using BindGroupHashMap = HashMap<uint32_t, RefPtr<const BindGroup>, DefaultHash<uint32_t>, WTF::UnsignedWithZeroKeyHashTraits<uint32_t>>;
    NSString* errorValidatingBindGroupCompatibility(const BindGroupHashMap&) const;
    uint64_t uniqueId() const { return m_uniqueId; }
private:
    PipelineLayout(std::optional<Vector<Ref<BindGroupLayout>>>&&, bool, uint64_t uniqueId, Device&);
    PipelineLayout(Device&);

    using DynamicOffsetMapValue = std::optional<uint32_t>;
    DynamicOffsetMapValue vertexOffsetForBindGroup(uint32_t) const;
    DynamicOffsetMapValue fragmentOffsetForBindGroup(uint32_t) const;
    DynamicOffsetMapValue computeOffsetForBindGroup(uint32_t) const;

    std::optional<Vector<Ref<BindGroupLayout>>> m_bindGroupLayouts;
    using DynamicOffsetMap = HashMap<uint32_t, DynamicOffsetMapValue, DefaultHash<uint32_t>, WTF::UnsignedWithZeroKeyHashTraits<uint32_t>>;
    DynamicOffsetMap m_vertexDynamicOffsets;
    DynamicOffsetMap m_fragmentDynamicOffsets;
    DynamicOffsetMap m_computeDynamicOffsets;

    using DynamicOffsetBufferMap = HashMap<uint32_t, uint64_t, DefaultHash<uint32_t>, WTF::UnsignedWithZeroKeyHashTraits<uint32_t>>;
    mutable DynamicOffsetBufferMap m_vertexOffsets;
    mutable DynamicOffsetBufferMap m_fragmentOffsets;
    mutable DynamicOffsetBufferMap m_computeOffsets;
    bool offsetVectorForBindGroup(uint32_t bindGroupIndex, DynamicOffsetBufferMap& stageOffsets, const Vector<uint32_t>& dynamicOffsets, WGPUShaderStage, std::span<uint32_t> destination);


    const Ref<Device> m_device;
    uint64_t m_uniqueId { 0 };
    using LinearLayoutOffsetContainer = HashMap<uint32_t, std::pair<uint32_t, uint32_t>, DefaultHash<uint32_t>, WTF::UnsignedWithZeroKeyHashTraits<uint32_t>>;
    EnumeratedArray<ShaderStage, LinearLayoutOffsetContainer, ShaderStage::Compute> m_linearLayoutOffset;
    bool m_isValid { true };
    const bool m_isAutogenerated { false };
};

} // namespace WebGPU
