#![deny(missing_docs)]
//! Generate the POSIX shell test suite for remrun.
//!
//! Read the JSON test definitions and convert them to a POSIX shell program.
/*
 * Copyright (c) 2022  Peter Pentchev <roam@ringlet.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

use std::fs;

use anyhow::{Context, Result};
use config_diag::ConfigDiag;
use itertools::Itertools;

mod builder;
mod cli;
mod defs;
mod loader;

fn main() -> Result<()> {
    let cfg = cli::parse_args().context("Could not parse the command-line options")?;
    cfg.diag_("starting up");
    let test_defs = loader::load_defs(&cfg).context("Could not load the test definitions")?;
    cfg.diag(|| format!("loaded {} tests", test_defs.tests.len()));
    cfg.diag(|| {
        test_defs
            .tests
            .iter()
            .map(|tdef| format!("- {}", tdef.title))
            .join("\n")
    });

    let contents = builder::generate(&cfg, &test_defs)
        .context("Could not generate the test file's contents")?;
    cfg.diag(|| {
        format!(
            "Writing {} bytes of output to {}",
            contents.len(),
            cfg.output_file.display()
        )
    });

    fs::write(&cfg.output_file, &contents).with_context(|| {
        format!(
            "Could not write {} bytes to {}",
            contents.len(),
            cfg.output_file.display()
        )
    })?;
    Ok(())
}
